const express = require('express');
const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
const { PrismaClient } = require('@prisma/client');
const { requireAuth } = require('../middleware/auth');

const router = express.Router();
const prisma = new PrismaClient();

// Create checkout session
router.post('/create-checkout-session', requireAuth, async (req, res) => {
  try {
    const { plan } = req.body; // 'plus' or 'pro'
    
    if (!plan || !['plus', 'pro'].includes(plan)) {
      return res.status(400).json({ error: 'Invalid plan selected' });
    }
    
    const user = await prisma.user.findUnique({
      where: { id: req.session.userId }
    });
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    // Create or retrieve customer
    let customerId = user.customerId;
    
    if (!customerId) {
      const customer = await stripe.customers.create({
        email: user.email,
        metadata: {
          userId: user.id
        }
      });
      
      customerId = customer.id;
      
      await prisma.user.update({
        where: { id: user.id },
        data: { customerId }
      });
    }
    
    // Select correct price ID based on plan
    const priceId = plan === 'plus' 
      ? process.env.STRIPE_PLUS_PRICE_ID 
      : process.env.STRIPE_PRO_PRICE_ID;
    
    // Create checkout session
    const session = await stripe.checkout.sessions.create({
      customer: customerId,
      payment_method_types: ['card'],
      line_items: [
        {
          price: priceId,
          quantity: 1,
        },
      ],
      mode: 'subscription',
      success_url: `${req.headers.origin}/dashboard?success=true`,
      cancel_url: `${req.headers.origin}/dashboard?canceled=true`,
      metadata: {
        userId: user.id,
        planType: plan
      }
    });
    
    res.json({ sessionId: session.id, url: session.url });
  } catch (error) {
    console.error('Stripe checkout error:', error);
    res.status(500).json({ error: 'Failed to create checkout session' });
  }
});

// Stripe webhook
// Note: raw body is handled in server.js
router.post('/webhook', async (req, res) => {
  const sig = req.headers['stripe-signature'];
  let event;
  
  try {
    event = stripe.webhooks.constructEvent(
      req.body,
      sig,
      process.env.STRIPE_WEBHOOK_SECRET
    );
  } catch (err) {
    console.error('Webhook signature verification failed:', err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }
  
  try {
    switch (event.type) {
      case 'checkout.session.completed': {
        const session = event.data.object;
        const userId = session.metadata.userId;
        const planType = session.metadata.planType; // 'plus' or 'pro'
        const subscriptionId = session.subscription;
        
        // Calculate next reset date (30 days from subscription date)
        const subscriptionDate = new Date();
        const nextResetDate = new Date(subscriptionDate);
        nextResetDate.setDate(nextResetDate.getDate() + 30); // 30 days from now
        
        const updateData = {
          subscriptionId,
          subscriptionStatus: 'active',
          planType: planType
        };
        
        // Set credits based on plan
        if (planType === 'plus') {
          updateData.credits = 1000;
          updateData.monthlyCredits = 1000;
          updateData.creditsResetDate = nextResetDate;
        } else if (planType === 'pro') {
          updateData.credits = 999999; // Effectively unlimited
          updateData.monthlyCredits = null;
          updateData.creditsResetDate = null;
        }
        
        await prisma.user.update({
          where: { id: userId },
          data: updateData
        });
        
        console.log(`✅ ${planType.toUpperCase()} subscription activated for user ${userId}`);
        console.log(`   Credits reset date: ${nextResetDate.toISOString()}`);
        break;
      }
      
      case 'customer.subscription.updated': {
        const subscription = event.data.object;
        
        const user = await prisma.user.findFirst({
          where: { subscriptionId: subscription.id }
        });
        
        if (user) {
          await prisma.user.update({
            where: { id: user.id },
            data: {
              subscriptionStatus: subscription.status
            }
          });
        }
        
        console.log(`✅ Subscription updated: ${subscription.status}`);
        break;
      }
      
      case 'customer.subscription.deleted': {
        const subscription = event.data.object;
        
        const user = await prisma.user.findFirst({
          where: { subscriptionId: subscription.id }
        });
        
        if (user) {
          await prisma.user.update({
            where: { id: user.id },
            data: {
              subscriptionStatus: 'canceled'
            }
          });
        }
        
        console.log(`✅ Subscription canceled`);
        break;
      }
      
      default:
        console.log(`Unhandled event type: ${event.type}`);
    }
    
    res.json({ received: true });
  } catch (error) {
    console.error('Webhook processing error:', error);
    res.status(500).json({ error: 'Webhook processing failed' });
  }
});

// Get subscription status
router.get('/subscription-status', requireAuth, async (req, res) => {
  try {
    const user = await prisma.user.findUnique({
      where: { id: req.session.userId },
      select: {
        subscriptionStatus: true,
        subscriptionId: true
      }
    });
    
    res.json({
      subscriptionStatus: user.subscriptionStatus,
      hasActiveSubscription: user.subscriptionStatus === 'active'
    });
  } catch (error) {
    console.error('Subscription status error:', error);
    res.status(500).json({ error: 'Failed to get subscription status' });
  }
});

module.exports = router;

