const express = require('express');
const { PrismaClient } = require('@prisma/client');
const { requireAuth, requireCredits } = require('../middleware/auth');
const { generateContent } = require('../services/openai');

const router = express.Router();
const prisma = new PrismaClient();

// Generate content
router.post('/generate', requireAuth, requireCredits, async (req, res) => {
  try {
    const { type, prompt, niche, tone, platform } = req.body;
    
    if (!type || !prompt) {
      return res.status(400).json({ error: 'Type and prompt are required' });
    }
    
    const user = req.user;
    
    // Generate content using OpenAI
    const result = await generateContent(type, { prompt, niche, tone, platform });
    
    // Save generation to database
    const generation = await prisma.generation.create({
      data: {
        userId: user.id,
        type,
        prompt,
        result
      }
    });
    
    // Deduct credit only for Free and Plus users (not Pro)
    if (user.planType !== 'pro') {
      await prisma.user.update({
        where: { id: user.id },
        data: { credits: user.credits - 1 }
      });
    }
    
    // Calculate remaining credits based on plan
    let remainingCredits;
    if (user.planType === 'pro') {
      remainingCredits = 'unlimited';
    } else if (user.planType === 'plus' || user.planType === 'free') {
      remainingCredits = user.credits - 1;
    }
    
    res.json({
      success: true,
      generation: {
        id: generation.id,
        type: generation.type,
        result: generation.result,
        createdAt: generation.createdAt
      },
      remainingCredits: remainingCredits
    });
  } catch (error) {
    console.error('Generation error:', error);
    res.status(500).json({ error: 'Failed to generate content' });
  }
});

// Get user's generation history
router.get('/history', requireAuth, async (req, res) => {
  try {
    const { limit = 20, offset = 0 } = req.query;
    
    const generations = await prisma.generation.findMany({
      where: { userId: req.session.userId },
      orderBy: { createdAt: 'desc' },
      take: parseInt(limit),
      skip: parseInt(offset),
      select: {
        id: true,
        type: true,
        prompt: true,
        result: true,
        createdAt: true
      }
    });
    
    const total = await prisma.generation.count({
      where: { userId: req.session.userId }
    });
    
    res.json({
      generations,
      total,
      limit: parseInt(limit),
      offset: parseInt(offset)
    });
  } catch (error) {
    console.error('History error:', error);
    res.status(500).json({ error: 'Failed to fetch history' });
  }
});

// Get single generation
router.get('/history/:id', requireAuth, async (req, res) => {
  try {
    const generation = await prisma.generation.findFirst({
      where: {
        id: req.params.id,
        userId: req.session.userId
      }
    });
    
    if (!generation) {
      return res.status(404).json({ error: 'Generation not found' });
    }
    
    res.json({ generation });
  } catch (error) {
    console.error('Get generation error:', error);
    res.status(500).json({ error: 'Failed to fetch generation' });
  }
});

module.exports = router;

