// Reusable UI Components

// Toast notification component
function showToast(message, type = 'success') {
  const toastId = 'toast-' + Date.now();
  const bgColor = type === 'success' ? 'bg-green-500' : type === 'error' ? 'bg-red-500' : 'bg-blue-500';
  
  const toast = `
    <div id="${toastId}" class="toast ${bgColor} text-white px-6 py-4 rounded-lg shadow-lg flex items-center space-x-3">
      <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        ${type === 'success' ? '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>' : '<path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"></path>'}
      </svg>
      <span>${message}</span>
    </div>
  `;
  
  $('body').append(toast);
  
  setTimeout(() => {
    $('#' + toastId).fadeOut(300, function() {
      $(this).remove();
    });
  }, 3000);
}

// Loading spinner component
function showLoading(elementId) {
  $(`#${elementId}`).html(`
    <div class="flex justify-center items-center py-8">
      <div class="spinner"></div>
    </div>
  `);
}

// Modal component
function showModal(title, content, actions = '') {
  const modalId = 'modal-' + Date.now();
  
  const modal = `
    <div id="${modalId}" class="fixed inset-0 z-50 overflow-y-auto modal-backdrop" aria-labelledby="modal-title" role="dialog" aria-modal="true">
      <div class="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
        <div class="fixed inset-0 transition-opacity" aria-hidden="true">
          <div class="absolute inset-0 bg-gray-500 opacity-75"></div>
        </div>
        
        <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
        
        <div class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full fade-in">
          <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
            <div class="sm:flex sm:items-start">
              <div class="mt-3 text-center sm:mt-0 sm:text-left w-full">
                <h3 class="text-2xl leading-6 font-bold text-gray-900 mb-4" id="modal-title">
                  ${title}
                </h3>
                <div class="mt-2">
                  ${content}
                </div>
              </div>
            </div>
          </div>
          <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
            ${actions || '<button onclick="closeModal(\'' + modalId + '\')" class="w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm">Close</button>'}
          </div>
        </div>
      </div>
    </div>
  `;
  
  $('body').append(modal);
  return modalId;
}

function closeModal(modalId) {
  $('#' + modalId).fadeOut(300, function() {
    $(this).remove();
  });
}

// Upgrade modal with plan selection
function showUpgradeModal() {
  const content = `
    <div class="text-center">
      <div class="mb-4">
        <svg class="mx-auto h-16 w-16 text-yellow-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"></path>
        </svg>
      </div>
      <p class="text-lg text-gray-700 mb-4">You've run out of credits!</p>
      <p class="text-gray-600 mb-6">Choose a plan to continue creating amazing content</p>
      
      <!-- Plus Plan -->
      <div class="bg-gradient-to-r from-blue-100 to-indigo-100 rounded-lg p-6 mb-4">
        <h4 class="font-bold text-2xl mb-2">$10<span class="text-lg font-normal">/month</span></h4>
        <h5 class="font-semibold text-lg mb-3">Plus Plan</h5>
        <ul class="text-left space-y-2 text-gray-700 text-sm mb-4">
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            1000 credits per month
          </li>
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            Resets monthly
          </li>
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            All content types
          </li>
        </ul>
        <button onclick="handleUpgrade('plus')" class="w-full bg-blue-600 hover:bg-blue-700 text-white px-4 py-2 rounded-lg font-semibold transition">
          Choose Plus
        </button>
      </div>
      
      <!-- Pro Plan -->
      <div class="bg-gradient-to-r from-purple-100 to-indigo-100 rounded-lg p-6 mb-4 border-2 border-purple-500">
        <div class="absolute top-0 right-0 bg-yellow-400 text-gray-900 px-3 py-1 rounded-bl-lg text-xs font-bold">
          BEST VALUE
        </div>
        <h4 class="font-bold text-2xl mb-2">$15<span class="text-lg font-normal">/month</span></h4>
        <h5 class="font-semibold text-lg mb-3">Pro Plan</h5>
        <ul class="text-left space-y-2 text-gray-700 text-sm mb-4">
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            <strong>Unlimited</strong> generations
          </li>
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            All content types
          </li>
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            Priority support
          </li>
          <li class="flex items-center">
            <svg class="w-5 h-5 text-green-500 mr-2 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
              <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
            </svg>
            Cancel anytime
          </li>
        </ul>
        <button onclick="handleUpgrade('pro')" class="w-full btn-gradient text-white px-4 py-2 rounded-lg font-semibold">
          Choose Pro
        </button>
      </div>
    </div>
  `;
  
  const actions = `
    <button onclick="closeModal('upgrade-modal')" class="w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 focus:outline-none sm:w-auto sm:text-sm">
      Maybe Later
    </button>
  `;
  
  const modalId = showModal('Choose Your Plan', content, actions);
  $('#' + modalId).attr('id', 'upgrade-modal');
}

// Handle upgrade button click with plan selection
function handleUpgrade(plan) {
  if (!plan || !['plus', 'pro'].includes(plan)) {
    showToast('Please select a valid plan', 'error');
    return;
  }
  
  $.ajax({
    url: '/api/stripe/create-checkout-session',
    method: 'POST',
    contentType: 'application/json',
    data: JSON.stringify({ plan: plan }),
    success: function(response) {
      if (response.url) {
        window.location.href = response.url;
      }
    },
    error: function(xhr) {
      showToast('Failed to start checkout. Please try again.', 'error');
    }
  });
}

// Credit badge component
function getCreditBadge(credits, planType) {
  if (planType === 'pro') {
    return '<span class="credit-badge">✨ Unlimited</span>';
  }
  
  const color = credits > 5 ? 'bg-green-500' : credits > 0 ? 'bg-yellow-500' : 'bg-red-500';
  return `<span class="credit-badge ${color}">${credits} Credits</span>`;
}

// Get plan name
function getPlanName(planType) {
  const plans = {
    'free': 'Free Plan',
    'plus': 'Plus Plan',
    'pro': 'Pro Plan'
  };
  return plans[planType] || 'Free Plan';
}

// Get plan badge
function getPlanBadge(planType) {
  const badges = {
    'free': '<span class="px-3 py-1 rounded-full text-sm font-semibold bg-gray-200 text-gray-800">Free</span>',
    'plus': '<span class="px-3 py-1 rounded-full text-sm font-semibold bg-blue-500 text-white">Plus</span>',
    'pro': '<span class="px-3 py-1 rounded-full text-sm font-semibold bg-gradient-to-r from-purple-500 to-indigo-600 text-white">Pro</span>'
  };
  return badges[planType] || badges['free'];
}

// Format date
function formatDate(dateString) {
  const date = new Date(dateString);
  const now = new Date();
  const diff = now - date;
  const days = Math.floor(diff / (1000 * 60 * 60 * 24));
  
  if (days === 0) {
    const hours = Math.floor(diff / (1000 * 60 * 60));
    if (hours === 0) {
      const minutes = Math.floor(diff / (1000 * 60));
      return `${minutes} minute${minutes !== 1 ? 's' : ''} ago`;
    }
    return `${hours} hour${hours !== 1 ? 's' : ''} ago`;
  } else if (days === 1) {
    return 'Yesterday';
  } else if (days < 7) {
    return `${days} days ago`;
  } else {
    return date.toLocaleDateString('en-US', { month: 'short', day: 'numeric', year: 'numeric' });
  }
}

// Content type label
function getContentTypeLabel(type) {
  const labels = {
    instagram_caption: '📸 Instagram Caption',
    tiktok_idea: '🎵 TikTok Idea',
    reel_script: '🎬 Reel Script',
    content_plan: '📅 Content Plan',
    quote_hook: '💭 Quote/Hook'
  };
  return labels[type] || type;
}

// Copy to clipboard
function copyToClipboard(text, elementId) {
  const textarea = document.createElement('textarea');
  textarea.value = text;
  document.body.appendChild(textarea);
  textarea.select();
  document.execCommand('copy');
  document.body.removeChild(textarea);
  
  showToast('Copied to clipboard!', 'success');
  
  // Visual feedback
  if (elementId) {
    const originalText = $(`#${elementId}`).text();
    $(`#${elementId}`).text('Copied!');
    setTimeout(() => {
      $(`#${elementId}`).text(originalText);
    }, 1000);
  }
}

