// Authentication middleware
function requireAuth(req, res, next) {
  if (!req.session || !req.session.userId) {
    return res.status(401).json({ error: 'Unauthorized. Please log in.' });
  }
  next();
}

// Check if user has credits or active subscription
async function requireCredits(req, res, next) {
  try {
    const { PrismaClient } = require('@prisma/client');
    const prisma = new PrismaClient();
    
    let user = await prisma.user.findUnique({
      where: { id: req.session.userId }
    });
    
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }
    
    // Check if Plus user needs credit reset
    if (user.planType === 'plus' && user.creditsResetDate) {
      const now = new Date();
      if (now >= new Date(user.creditsResetDate)) {
        // Reset credits to monthly limit
        // Set next reset to 30 days from today (maintaining the billing cycle)
        const nextResetDate = new Date(now);
        nextResetDate.setDate(nextResetDate.getDate() + 30);
        
        user = await prisma.user.update({
          where: { id: user.id },
          data: {
            credits: user.monthlyCredits || 1000,
            creditsResetDate: nextResetDate
          }
        });
        console.log(`✅ Credits reset for Plus user ${user.email}`);
        console.log(`   Next reset: ${nextResetDate.toISOString()}`);
      }
    }
    
    // Pro users have unlimited (effectively)
    if (user.planType === 'pro' && user.subscriptionStatus === 'active') {
      req.user = user;
      return next();
    }
    
    // Plus and Free users need to check credits
    if (user.credits <= 0) {
      return res.status(403).json({ 
        error: 'No credits remaining', 
        needsUpgrade: true 
      });
    }
    
    req.user = user;
    next();
  } catch (error) {
    console.error('Auth middleware error:', error);
    res.status(500).json({ error: 'Server error' });
  }
}

module.exports = {
  requireAuth,
  requireCredits
};

